"""Map APIs."""

from typing import List

from fastapi import (
    APIRouter,
    HTTPException,
    Request,
)

from app.api.v1.auth import get_current_user
from app.core.config import settings
from app.core.limiter import limiter
from app.core.logging import logger
from app.schemas.map import MapResponse
from app.services.database import DatabaseService
from app.utils.sanitization import sanitize_string

router = APIRouter()
db_service = DatabaseService()


@router.get("/provinces", response_model=List[MapResponse])
# @limiter.limit(settings.RATE_LIMIT_ENDPOINTS["messages"][0])
async def get_maps(
    request: Request,
):
    """Get all provinces info.

    Args:
        request: The FastAPI request object for rate limiting.

    Returns:
        List[MapResponse]: List of provinces info.

    Raises:
        HTTPException: If there's an error retrieving the map.
    """
    try:
        maps = await db_service.get_map()
        return [
            MapResponse(
                id=sanitize_string(map_item.id),
                province=sanitize_string(map_item.province),
                active_user=sanitize_string(map_item.active_user),
                related_news=sanitize_string(map_item.related_news),
                tags=map_item.tags,
                agent_analysis=sanitize_string(map_item.agent_analysis),
            )
            for map_item in maps
        ]
    except ValueError as ve:
        logger.error("get_map_validation_failed", error=str(ve), exc_info=True)
        raise HTTPException(status_code=422, detail=str(ve))
